```markdown
# Vol 4: Continuum Hamiltonian → Path Integral Automation

This repository implements the Volume 4 pipeline of the Absolute Relativity (AR) series: recovering a continuum quantum‑field action from discrete fractal‑gauge data, sampling it via Metropolis–Hastings, and extracting key physics observables (correlators, β‑functions, anomalies, composite moments, RG flow).

---

## Repository Structure

```

vol-4-hamiltonian-path-integral/
├── README.md
├── LICENSE
├── envs/
│   └── pipeline.yaml          # Conda/Mamba environment spec (includes editable install)
├── data/
│   ├── D\_values.csv           # Fractal‑dimension anchor data
│   ├── fractal\_fit.pkl        # (auto) logistic‑fit parameters
│   ├── pivot\_params.json      # (auto) pivot a, b
│   ├── kernel.npy             # (auto) reproduction‑kernel matrix
│   └── discrete\_action.pkl    # (auto) symbolic S\_discrete
├── ar\_sim/                    # Core library
│   ├── common/
│   │   ├── fractal\_fits.py
│   │   ├── kernel\_builder.py
│   │   └── composite\_moment.py
│   ├── continuum\_action.py
│   └── path\_integral.py
├── scripts/                   # CLI wrappers
│   ├── fit\_fractal.py
│   ├── calibrate\_pivot.py
│   ├── build\_kernel.py
│   ├── compute\_action.py
│   ├── run\_path\_integral.py
│   ├── analyze\_observables.py
│   ├── compute\_anomalies.py
│   ├── apply\_composite\_moment.py
│   └── run\_rg\_flow\.py
├── analysis/                  # Analysis modules & entry points
│   ├── **init**.py
│   └── observables.py
├── results/                   # (auto) HDF5 & NumPy outputs
│   ├── samples.h5
│   ├── anomalies.npy
│   ├── composite.h5
│   └── rg\_flow\.npy
├── analysis/                  # (auto) .npy observables
│   ├── correlator.npy
│   └── beta.npy
├── logs/                      # (optional) per‑rule logfiles
├── tests/                     # Unit tests
│   ├── test\_fractal\_fits.py
│   └── test\_kernel\_spectra.py
└── PROVENANCE.md              # Data origins & commit‑hash records

````

---

## Getting Started

### 1. Clone & enter the repo

```bash
git clone <repo-url> vol-4-hamiltonian-path-integral
cd vol-4-hamiltonian-path-integral
````

### 2. Create & activate the environment

The `envs/pipeline.yaml` spec installs all dependencies **and** performs an editable install of this repo so that `import ar_sim` and `import analysis` work immediately.

```bash
# If you have mamba:
mamba env create -f envs/pipeline.yaml -n vol4
# Otherwise with conda:
conda env create -f envs/pipeline.yaml -n vol4

conda activate vol4
```

### 3. (Optional) Inspect or modify code

All core modules live under `ar_sim/` and `analysis/`.
Scripts in `scripts/` wrap those modules for Snakemake.

### 4. Validate the workflow

Lint your Snakefile and preview the DAG without running heavy steps:

```bash
snakemake --lint
snakemake --use-conda --cores 2 --dry-run
```

### 5. Run the full pipeline

This will sequentially execute every step—from fractal fit → pivot calibration → kernel build → path integral → observables → anomalies → composite moments → RG flow:

```bash
snakemake --use-conda --cores 2
```

At the end you will have:

* **data/**: `fractal_fit.pkl`, `pivot_params.json`, `kernel.npy`, `discrete_action.pkl`
* **results/**: `samples.h5`, `anomalies.npy`, `composite.h5`, `rg_flow.npy`
* **analysis/**: `correlator.npy`, `beta.npy`

---

## Inspecting Your Results

You can load and plot the outputs in Python or a Jupyter notebook. For example:

```python
import numpy as np
import matplotlib.pyplot as plt

beta = np.load("analysis/beta.npy")
plt.plot(beta, marker="o")
plt.xlabel("Index"); plt.ylabel("β‑function")
plt.title("Estimated β‑function from MCMC samples")
plt.show()
```

---

## Running Tests

Unit tests cover your fractal‑fit and kernel routines:

```bash
pytest -q
```

---

## Freezing the Release

Once you are satisfied:

1. **Delete** any untracked logs or old outputs.
2. **Commit** your final changes:

   ```bash
   git add .
   git commit -m "Release: Volume 4 pipeline complete"
   ```
3. **Push** to GitHub and tag:

   ```bash
   git push origin main
   git tag -a v4.0.0 -m "AR Vol 4 release"
   git push origin v4.0.0
   ```

You can now fork or archive this repo as your immutable Vol 4 release.

---

## License

This project is licensed under the MIT License. See [LICENSE](LICENSE) for details.
